import React, { useEffect, useState } from "react";
import {
  View,
  Text,
  ScrollView,
  Linking,
  TouchableOpacity,
} from "react-native";
import { collection, getDocs, doc, getDoc } from "firebase/firestore";
import { auth, db } from "../../config/firebase";
import moment from "moment";
import Loader from "../components/Loaders/Loader";

const ShowComments = () => {
  const [bugs, setBugs] = useState([]);
  const [loading, setLoading] = useState(true);

  // Fonction pour récupérer les bugs et les emails des utilisateurs
  const fetchBugsWithUsers = async () => {
    try {
      const bugsCollection = collection(db, "comments");
      const bugSnapshot = await getDocs(bugsCollection);
      const bugsWithUsers = await Promise.all(
        bugSnapshot.docs.map(async (bugDoc) => {
          const bugData = bugDoc.data();
          const userId = bugData.userId;

          // Récupérez l'email de l'utilisateur
          let userEmail = "Email inconnu";
          if (userId) {
            const userDocRef = doc(db, "users", userId);
            const userDoc = await getDoc(userDocRef);
            if (userDoc.exists()) {
              userEmail = userDoc.data().email || "Email non spécifié";
            }
          }

          return {
            id: bugDoc.id,
            ...bugData,
            userEmail,
          };
        })
      );
      setBugs(bugsWithUsers);
      setLoading(false);
    } catch (error) {
      console.error("Erreur lors de la récupération des bugs:", error);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchBugsWithUsers();
  }, []);

  return (
    <ScrollView className="flex-1 bg-gray-100 p-4">
      <Text className="text-2xl font-bold text-center mb-4">
        Liste des commentaires
      </Text>
      {loading ? (
        <Loader />
      ) : bugs.length === 0 ? (
        <Text className="text-center text-gray-500">Aucun bug trouvé.</Text>
      ) : (
        bugs.map((bug) => (
          <View key={bug.id} className="bg-white rounded-lg shadow-md p-4 mb-4">
            <Text className="text-lg font-semibold">
              {bug.userEmail || "Sans titre"}
            </Text>
            <Text className="text-gray-600">
              {bug.comment || "Pas de description"}
            </Text>
            <Text className="text-sm text-gray-400">
              Date :{" "}
              {moment(bug.createdAt).format("DD/MM/YY") || "Non spécifiée"}
            </Text>
            <TouchableOpacity
              onPress={() => {
                const subject = "Réponse: Commentaire";
                const body = `Bonjour,

Voici une réponse à votre commentaire :

${bug.comment || ""}`;
                const mailto = `mailto:${
                  bug.userEmail
                }?subject=${encodeURIComponent(
                  subject
                )}&body=${encodeURIComponent(body)}`;
                Linking.openURL(mailto).catch((err) =>
                  console.error("Failed to open email client:", err)
                );
              }}
              className="bg-blue-500 text-white text-center rounded-lg mt-2 p-2"
            >
              <Text className="text-white font-semibold">Répondre</Text>
            </TouchableOpacity>
          </View>
        ))
      )}
    </ScrollView>
  );
};

export default ShowComments;
