import { View, Text } from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import { auth, db } from "../../../config/firebase";
import {
  addDoc,
  collection,
  doc,
  getDoc,
  getDocs,
  limit,
  onSnapshot,
  orderBy,
  query,
  startAfter,
  updateDoc,
  where,
} from "firebase/firestore";
import { Pressable } from "react-native";
import { SafeAreaView } from "react-native";
import { FlatList } from "react-native";
import moment from "moment";
import { COLORS } from "../../styles/colors";
import { TouchableOpacity } from "react-native";
import { Ionicons } from "@expo/vector-icons";
import { Image } from "react-native";
import Modal from "react-native-modal";
import {
  BUTTON_CLASS,
  BUTTON_OUTLINE_CLASS,
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  LISTING_LIMIT,
} from "../../styles/constants";
import PageLoader from "../../components/Loaders/PageLoader";
import SmallPrimaryButton from "../../components/Buttons/SmallPrimaryButton";
import PrimaryButton from "../../components/Buttons/PrimaryButton";
import { FlashList } from "@shopify/flash-list";
import ListIcons from "../../components/ListIcons";
import Loader from "../../components/Loaders/Loader";

export default function Users({ navigation }) {
  const [lastVisible, setLastVisible] = useState(null);
  const [hasMore, setHasMore] = useState(true);

  const [data, setData] = useState([]);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    getData();
  }, []);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerTitle: "Utilisateurs",
      headerRight: () => (
        <TouchableOpacity
          onPress={() => navigation.jumpTo("SearchUserTab")}
          className="items-center justify-center w-8 h-8 bg-white rounded-full"
        >
          <Ionicons name={`search-outline`} size={27} color={"black"} />
        </TouchableOpacity>
      ),
    });
  }, [navigation]);

  async function getData(lastDoc) {
    try {
      let q = query(
        collection(db, "users"),
        orderBy("createdAt", "desc"),
        limit(LISTING_LIMIT)
      );
      if (lastDoc) {
        q = query(q, startAfter(lastDoc));
      }
      const querySnapshot = await getDocs(q);
      const dataArr = [];
      let lastVisibleDoc = null;

      querySnapshot.forEach(async (res) => {
        dataArr.push({
          id: res.id,
          ...res.data(),
        });
        lastVisibleDoc = res;
      });
      if (lastDoc) {
        setData((prevData) => [...prevData, ...dataArr]);
      } else {
        setData(dataArr);
      }

      setLastVisible(lastVisibleDoc);
      setHasMore(dataArr.length === LISTING_LIMIT);
    } catch (error) {
      console.log(error);
    } finally {
      setIsLoading(false);
    }
  }

  const updateIsActive = async (value, itemId) => {
    const docRef = doc(db, "users", itemId);
    await updateDoc(docRef, {
      isActive: !value,
    });
    getData();
  };

  const renderFooter = () => {
    if (!isLoading) return null;
    return (
      <View className="pt-5">
        <Loader />
      </View>
    );
  };

  const handleLoadMore = () => {
    if (!isLoading) {
      getData(lastVisible);
    }
  };

  const Card = ({ item }) => (
    <Pressable
      className="p-3 bg-white border border-gray-200 mt-2 rounded-md flex-row items-center justify-between"
      onPress={() => navigation.navigate("UserDetails", { item })}
    >
      <View className="flex-1 flex-row items-center">
        <View>
          <View className="flex-row items-center">
            <Text
              className="text-base text-gray-700"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Nom: {item?.lastName.toUpperCase()}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-base text-gray-700 capitalize"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Prénom: {item?.firstName}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-base text-gray-700"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Compte vérifié: {item?.verification?.isVerified ? "Oui" : "Non"}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-xs text-blue-600"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Incrit le {moment(item?.createdAt).format("DD/MM/YYYY")} ·{" "}
              {item?.isActive == undefined
                ? "Actif"
                : item.isActive
                ? "Actif"
                : "Inactif"}
            </Text>
          </View>
        </View>
      </View>
      <View className="">
        <ListIcons
          notif
          onPressNotif={() =>
            navigation.navigate("SendNotifications", { item })
          }
          three
          onPress3={() =>
            updateIsActive(
              item.isActive == undefined ? true : item.isActive ? true : false,
              item.id
            )
          }
          itemIsActive={
            item.isActive == undefined ? true : item.isActive ? true : false
          }
        />
      </View>
    </Pressable>
  );

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-gray-100">
      <FlashList
        estimatedItemSize={50}
        className="p-4"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
        keyExtractor={(item) => item.id}
        data={data}
        renderItem={({ item }) => <Card item={item} />}
        ListFooterComponent={renderFooter}
        onEndReached={handleLoadMore}
        onEndReachedThreshold={0.1}
      />
    </SafeAreaView>
  );
}
