import React, { useState } from "react";
import {
  View,
  StyleSheet,
  TouchableOpacity,
  Text,
  Alert,
  Platform,
} from "react-native";
import { CardField, useConfirmPayment } from "@stripe/stripe-react-native";
import BackButton from "../components/Buttons/BackButton";
import { useLayoutEffect } from "react";
import { BUTTON_CLASS } from "../styles/constants";
import { addDoc, collection, doc, getDoc, updateDoc } from "firebase/firestore";
import { db } from "../../config/firebase";
import { SENDGRID_API_KEY, SENDGRID_FROM, BACKEND_API_URL } from "@env";
import axios from "axios";
import moment from "moment";

const Pay = ({ navigation, route }) => {
  const [cardDetails, setCardDetails] = useState();
  const { confirmPayment, loading } = useConfirmPayment();
  const { searchId, searchFinderId, amount } = route.params;

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: "Payer par Carte Bancaire",
    });
  });

  const fetchPaymentIntentClientSecret = async () => {
    const response = await fetch(`${BACKEND_API_URL}/create-payment-intent`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        amount: Math.round(amount),
        currency: "eur",
      }),
    });
    const { clientSecret } = await response.json();
    return clientSecret;
  };

  const handlePayPress = async () => {
    const clientSecret = await fetchPaymentIntentClientSecret();
    const { paymentIntent, error } = await confirmPayment(clientSecret, {
      paymentMethodType: "Card",
      paymentMethodData: {},
    });

    if (error) {
      console.log(error);
      Alert.alert(
        "Le paiement a échoué",
        "Nous vous prions de rééssayer ultérieurement"
      );
      // alert(`Payment Confirmation Error ${error.message}`);
    } else if (paymentIntent) {
      updateToIsPaied();
      Alert.alert(
        "Paiement réussi",
        "Vous pouvez maintenant consulter l'adresse de votre véhicule",
        [{ text: "Voir les infos", onPress: () => navigation.goBack() }],
        {
          cancelable: true,
        }
      );
    }
  };

  const updateToIsPaied = async () => {
    const docRef = doc(db, "searchs", searchId);
    await updateDoc(docRef, {
      paymentStatus: "paid",
      paymentMethod: "stripe",
      price: String(amount),
    });
    const getFinderInfos = await getDoc(doc(db, "users", searchFinderId));
    sendPushNotifToFinder(getFinderInfos.data().expoPushToken);
    sendMailToSearcher(getFinderInfos.data());
    sendNotifToFinder(searchFinderId);
  };

  function enleverEspaces(chaine) {
    return chaine.replace(/\s/g, "");
  }

  const sendMailToSearcher = async (finderInfo) => {
    await axios.post(
      "https://api.sendgrid.com/v3/mail/send",
      {
        personalizations: [
          {
            to: [
              {
                email: `${finderInfo.email}`,
              },
            ],
            dynamic_template_data: {
              first_name:
                finderInfo.firstName !== ""
                  ? finderInfo.firstName + " " + finderInfo.lastName
                  : "cher utilisateur",
            },
          },
        ],
        from: {
          email: `${SENDGRID_FROM}`,
        },
        template_id: enleverEspaces("d-930e11a826e145cba20f78914ae18841"),
      },
      {
        headers: {
          authorization: `Bearer ${SENDGRID_API_KEY}`,
          "Content-Type": "application/json",
        },
      }
    );
  };

  async function sendPushNotifToFinder(expoPushToken) {
    const message = {
      to: expoPushToken,
      sound: "default",
      title: "Vous avez un match ! ✅",
      body: "Votre signalement au sein de l'application Sur4plots a permis à un citoyen de retrouver son véhicule !",
    };

    await fetch("https://exp.host/--/api/v2/push/send", {
      method: "POST",
      headers: {
        Accept: "application/json",
        "Accept-encoding": "gzip, deflate",
        "Content-Type": "application/json",
      },
      body: JSON.stringify(message),
    });
  }

  async function sendNotifToFinder(searchFinderId) {
    const dataToAdd = {
      title: "Vous avez un match ! ✅",
      title_en: "You have a match! ✅",
      text: "Votre signalement au sein de l'application Sur4plots a permis à un citoyen de retrouver son véhicule !",
      text_en: "Your report within the Sur4plots application allowed a citizen to find his vehicle!",
      userId: searchFinderId,
      isViewed: false,
      createdAt: moment().format(),
    };

    await addDoc(collection(db, "notifications"), dataToAdd);
  }

  return (
    <View style={styles.container}>
      <CardField
        postalCodeEnabled={false}
        placeholder={{
          number: "4242 4242 4242 4242",
          // number: "XXXX XXXX XXXX XXXX",
          // cvc: "123",
          // expiration: "12/30",
        }}
        cardStyle={styles.card}
        style={styles.cardContainer}
        onCardChange={(cardDetails) => {
          setCardDetails(cardDetails);
        }}
      />
      <TouchableOpacity
        className={BUTTON_CLASS}
        disabled={loading}
        onPress={handlePayPress}
      >
        <Text
          className="text-sm text-white"
          style={{ fontFamily: "Inter_700Bold" }}
        >
          Payer {amount}€ maintenant
        </Text>
      </TouchableOpacity>
    </View>
  );
};
export default Pay;

const styles = StyleSheet.create({
  container: {
    // flex: 1,
    justifyContent: "center",
    margin: 20,
  },
  input: {
    backgroundColor: "#efefefef",
    borderRadius: 8,
    fontSize: 20,
    height: 50,
    padding: 10,
  },
  card: {
    backgroundColor: "#efefefef",
    textColor: Platform.OS == "ios" ? "#333" : null,
    placeholderColor: Platform.OS == "ios" ? "#a4b0be" : null,
    fontFamily: Platform.OS == "ios" ? "semibold" : null,
  },
  cardContainer: {
    height: 50,
    marginVertical: 30,
  },
});
