import {
  View,
  Text,
  SafeAreaView,
  TouchableOpacity,
  Image,
  Pressable,
  Alert,
} from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import { Ionicons } from "@expo/vector-icons";
import * as Haptics from "expo-haptics";
import BackButton from "../../../components/Buttons/BackButton";
import {
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  UPLOAD_PARAMS,
} from "../../../styles/constants";
import PageSubTitle from "../../../components/Titles/PageSubTitle";
import PrimaryButton from "../../../components/Buttons/PrimaryButton";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import * as ImageManipulator from "expo-image-manipulator";
import * as ImagePicker from "expo-image-picker";
import { auth, db } from "../../../../config/firebase";
import { getDownloadURL, ref, uploadBytesResumable } from "firebase/storage";
import { storage } from "../../../../config/firebase";
import { doc, onSnapshot, updateDoc } from "firebase/firestore";
import { sendMessageMail } from "../../../utils/sendMails";
import { imageCompressor, openSettings } from "../../../utils/allFunctions";
import TransparentLoader from "../../../components/Loaders/TransparentLoader";
import sendAdminNotif from "../../../utils/sendAdminNotif";

export default function ValidateProfileStep2({ navigation, route }) {
  const [isLoading, setIsLoading] = useState(false);
  const [photoIsLoading, setPhotoIsLoading] = useState(false);
  const [recto, setRecto] = useState(null);
  const [verso, setVerso] = useState(null);
  const [face, setFace] = useState(null);
  const [profile, setProfile] = useState(null);
  const [userInfo, setUserInfo] = useState(null);
  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: "Vérifier mes documents",
    });
  }, [navigation]);

  useEffect(() => {
    navigation
      .getParent()
      ?.setOptions({ tabBarStyle: { display: "none" }, tabBarVisible: false });
  }, [navigation]);

  useEffect(() => {
    if (!auth.currentUser) {
      setIsLoading(false);
      return;
    }
    getUserInfo();
  }, []);

  const getUserInfo = async () => {
    const unsub = onSnapshot(doc(db, "users", auth.currentUser.uid), (doc) => {
      setUserInfo({ id: doc.id, ...doc.data() });
    });
    return () => unsub();
  };

  const uploadPhotos = async () => {
    try {
      const imageUrls = [];
      const images = [recto, verso, face];

      for (const image of images) {
        const response = await fetch(image);
        const blobFile = await response.blob();

        const storageRef = ref(
          storage,
          `users_verification/${auth.currentUser.uid}` + "_" + Date.now()
        );

        const snapshot = await uploadBytesResumable(storageRef, blobFile);

        const imageUrl = await getDownloadURL(snapshot?.ref);
        imageUrls.push(imageUrl);
      }

      return imageUrls;
    } catch (error) {
      console.error("Error uploading or retrieving images:", error);
    }
  };

  const uploadProfilePicture = async () => {
    try {
      const response = await fetch(profile);

      const blobFile = await response.blob();
      const storageRef = ref(
        storage,
        `profile_pictures_/${auth.currentUser.uid}/` + "_" + Date.now()
      );

      const snapshot = await uploadBytesResumable(storageRef, blobFile);

      const imageUrl = await getDownloadURL(snapshot?.ref);
      return imageUrl;
    } catch (error) {
      console.error("Error uploading or retrieving image:", error);
    }
  };

  const validate = async () => {
    setIsLoading(true);

    if (!recto || !verso || !face || !profile) {
      Alert.alert(
        "Des photos sont manquantes",
        "Nous avons besoin du recto et du verso de votre pièce, le tout accompagné d'une photo claire et visible de votre visage et de votre photo de profil"
      );

      setIsLoading(false);
      return;
    }

    const profilPhotoUrl = await uploadProfilePicture();
    const imagesLinkArray = await uploadPhotos();

    const { data } = route.params;
    await updateDoc(doc(db, "users", auth.currentUser.uid), {
      verification: {
        type: data.type,
        idNumber: data.idNumber,
        photos: [imagesLinkArray[0], imagesLinkArray[1]],
        facePhoto: imagesLinkArray[2],
        isVerified: true,
      },
      profilPhotoUrl: profilPhotoUrl,
    });

    const messageToClient = {
      subject: "Compte vérifié ✅",
      title: "Votre compte est maintenant vérifié ✅",
      desc: `Si les informations fournies sont fausses/érronées, nous nous réservons le droit de suspendre votre compte à tout moment`,
    };

    // sendMessageMail(userInfo, messageToClient);
    sendAdminNotif(
      `${
        userInfo.firstName + " " + userInfo.lastName
      } vient de vous envoyer ses documents ✅`
    );
    Alert.alert(`${messageToClient.title}`, `${messageToClient.desc}`);
    setIsLoading(false);
    navigation.popToTop();
  };

  const pickImage = async (value) => {
    Alert.alert(
      "Choisir une option",
      "Veuillez sélectionner une option pour choisir une image",
      [
        {
          text: "Annuler",
          style: "cancel",
        },
        {
          text: "Caméra",
          onPress: async () => {
            const { status } =
              await ImagePicker.requestCameraPermissionsAsync();
            if (status !== "granted") {
              openSettings();
              return;
            }
            const result = await ImagePicker.launchCameraAsync({
              mediaTypes: ImagePicker.MediaTypeOptions.Images,
              aspect: UPLOAD_PARAMS.aspect,
              quality: UPLOAD_PARAMS.quality,
            });
            handleImagePickerResult(result, value);
          },
        },
        {
          text: "Bibliothèque",
          onPress: async () => {
            const { status } =
              await ImagePicker.requestMediaLibraryPermissionsAsync();
            if (status !== "granted") {
              openSettings();
              return;
            }
            const result = await ImagePicker.launchImageLibraryAsync({
              mediaTypes: ImagePicker.MediaTypeOptions.Images,
              aspect: UPLOAD_PARAMS.aspect,
              quality: UPLOAD_PARAMS.quality,
            });
            handleImagePickerResult(result, value);
          },
        },
      ],
      { cancelable: true }
    );
  };

  const cancelPhotoLoading = () => {
    setTimeout(() => {
      setPhotoIsLoading(false);
    }, 500);
  };

  const handleImagePickerResult = async (result, value) => {
    if (!result.canceled) {
      setPhotoIsLoading(true);
      const compressedImage = await imageCompressor(result.assets[0].uri);
      if (value == 1) {
        setRecto(compressedImage);
        cancelPhotoLoading();
      } else if (value == 2) {
        setVerso(compressedImage);
        cancelPhotoLoading();
      } else {
        cancelPhotoLoading();
        return;
      }
    }
  };

  const pickeFacePhoto = async () => {
    const { status } = await ImagePicker.requestCameraPermissionsAsync();
    if (status !== "granted") {
      openSettings();
      return;
    }

    const result = await ImagePicker.launchCameraAsync({
      mediaTypes: ImagePicker.MediaTypeOptions.Images,
      allowsEditing: false,
      aspect: UPLOAD_PARAMS.aspect,
      quality: UPLOAD_PARAMS.quality,
    });
    if (!result.canceled) {
      setPhotoIsLoading(true);
      const compressedImage = await imageCompressor(result.assets[0].uri);
      setFace(compressedImage);
      cancelPhotoLoading();
    }
  };

  const pickeProfilePhoto = async () => {
    Alert.alert(
      "Choisir une option",
      "Veuillez sélectionner une option pour choisir une image",
      [
        {
          text: "Annuler",
          style: "cancel",
        },
        {
          text: "Caméra",
          onPress: async () => {
            const { status } =
              await ImagePicker.requestCameraPermissionsAsync();
            if (status !== "granted") {
              openSettings();
              return;
            }
            const result = await ImagePicker.launchCameraAsync({
              mediaTypes: ImagePicker.MediaTypeOptions.Images,
              aspect: UPLOAD_PARAMS.aspect,
              quality: UPLOAD_PARAMS.quality,
            });
            if (!result.canceled) {
              setPhotoIsLoading(true);
              const compressedImage = await imageCompressor(
                result.assets[0].uri
              );
              setProfile(compressedImage);
              cancelPhotoLoading();
            }
          },
        },
        {
          text: "Bibliothèque",
          onPress: async () => {
            const { status } =
              await ImagePicker.requestMediaLibraryPermissionsAsync();
            if (status !== "granted") {
              openSettings();
              return;
            }
            const result = await ImagePicker.launchImageLibraryAsync({
              mediaTypes: ImagePicker.MediaTypeOptions.Images,
              aspect: UPLOAD_PARAMS.aspect,
              quality: UPLOAD_PARAMS.quality,
            });
            if (!result.canceled) {
              setPhotoIsLoading(true);
              const compressedImage = await imageCompressor(
                result.assets[0].uri
              );
              setProfile(compressedImage);
              cancelPhotoLoading();
            }
          },
        },
      ],
      { cancelable: true }
    );
  };

  const ImgCard = ({ img, type }) => (
    <View className="mt-4 relative w-[48%] h-[15vh]">
      <Pressable
        onPress={() => {
          if (type == 1) {
            pickImage(1);
          } else if (type == 2) {
            pickImage(2);
          } else {
            return;
          }
        }}
      >
        {img !== null ? (
          <Image
            source={{
              uri: img,
            }}
            resizeMode={"cover"}
            className="w-full h-full rounded-lg"
          />
        ) : (
          <View className="w-full h-full rounded-lg bg-gray-200 items-center justify-center">
            <Text
              className="text-sm text-center"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              {type == 1 ? "Recto\n(Devant)" : "Verso\n(Derrière)"}
            </Text>
          </View>
        )}
      </Pressable>
      {(recto || verso) && (
        <TouchableOpacity
          onPress={() => {
            if (type == 1) {
              setRecto(null);
            } else if (type == 2) {
              setVerso(null);
            } else {
              return;
            }
          }}
          className="bg-red-500 absolute bottom-2 right-2 w-7 h-7 rounded flex items-center justify-center"
        >
          <Ionicons name="trash" size={18} color="white" />
        </TouchableOpacity>
      )}
    </View>
  );

  const FaceCard = ({ img }) => (
    <View className="mt-4 relative w-[48%] h-[15vh]">
      <Pressable onPress={pickeFacePhoto}>
        {img !== null ? (
          <Image
            source={{
              uri: img,
            }}
            resizeMode={"cover"}
            className="w-full h-full rounded-lg"
          />
        ) : (
          <View className="w-full h-full rounded-lg bg-gray-200 items-center justify-center">
            <Text
              className="text-sm text-center"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Visage
            </Text>
          </View>
        )}
      </Pressable>
      {face && (
        <TouchableOpacity
          onPress={() => {
            setFace(null);
          }}
          className="bg-red-500 absolute bottom-2 right-2 w-7 h-7 rounded flex items-center justify-center"
        >
          <Ionicons name="trash" size={18} color="white" />
        </TouchableOpacity>
      )}
    </View>
  );

  const ProfileCard = ({ img }) => (
    <View className="mt-4 relative w-[48%] h-[15vh]">
      <Pressable onPress={pickeProfilePhoto}>
        {img !== null ? (
          <Image
            source={{
              uri: img,
            }}
            resizeMode={"cover"}
            className="w-full h-full rounded-lg"
          />
        ) : (
          <View className="w-full h-full rounded-lg bg-gray-200 items-center justify-center">
            <Text
              className="text-sm text-center"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Photo de profil
            </Text>
          </View>
        )}
      </Pressable>
      {profile && (
        <TouchableOpacity
          onPress={() => {
            setProfile(null);
          }}
          className="bg-red-500 absolute bottom-2 right-2 w-7 h-7 rounded flex items-center justify-center"
        >
          <Ionicons name="trash" size={18} color="white" />
        </TouchableOpacity>
      )}
    </View>
  );
  return (
    <View className="flex-1 bg-white">
      {photoIsLoading && <TransparentLoader />}
      <SafeAreaView className="flex-1 bg-white">
        <KeyboardAwareScrollView
          className="px-4"
          keyboardDismissMode="interactive"
          keyboardShouldPersistTaps="handled"
          // resetScrollToCoords={{ x: 0, y: 0 }}
          scrollEnabled
          behavior="padding"
          contentContainerStyle={{
            paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
          }}
        >
          <View className="mt-3">
            <PageSubTitle title={"Photo recto et verso de la pièce"} />

            <View className="flex-row flex-wrap justify-between">
              <ImgCard type={1} img={recto} />
              <ImgCard type={2} img={verso} />
            </View>
          </View>

          <View className="mt-3">
            <PageSubTitle title={"Photo du visage et de profil"} />

            <View className="flex-row flex-wrap justify-between">
              <FaceCard img={face} />
              <ProfileCard img={profile} />
            </View>
          </View>
          <Text
            className="mt-10 text-xs text-gray-400"
            style={{ fontFamily: "Inter_400Regular" }}
          >
            NB: Les photos de vos documents et de votre visage nous permettent
            de confirmer votre identité, nous ne les publierons jamais, la photo
            de profil quant à elle sera publiée sur votre profil.
          </Text>
          {/* reservation button */}
          <View className="flex-1 w-full mt-10">
            <PrimaryButton
              text={"Envoyer mes documents"}
              isLoading={isLoading}
              onPress={validate}
            />
          </View>
        </KeyboardAwareScrollView>
      </SafeAreaView>
    </View>
  );
}
