import { View, Text, SafeAreaView, ScrollView, Alert } from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import { useTranslation } from "react-i18next";
import BackButton from "../../components/Buttons/BackButton";
import { DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING, TEXT_AREA_CLASS } from "../../styles/constants";
import PrimaryButton from "../../components/Buttons/PrimaryButton";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import { TextInput } from "react-native";
import { addDoc, collection } from "firebase/firestore";
import { auth, db } from "../../../config/firebase";
import moment from "moment";

export default function WriteComment({ navigation }) {
  const { t } = useTranslation();
  const MAX_DESC_LENGTH = 500;
  const [isLoading, setIsLoading] = useState(false);
  const [desc, setDesc] = useState("");

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: t('write_comment.header_title'),
    });
  }, [navigation, t]);

  useEffect(() => {
    navigation
      .getParent()
      ?.setOptions({ tabBarStyle: { display: "none" }, tabBarVisible: false });
    return () =>
      navigation
        .getParent()
        ?.setOptions({ tabBarStyle: undefined, tabBarVisible: undefined });
  }, [navigation]);

  const validate = async () => {
    setIsLoading(true);

    if (desc.trim() === "") {
      setIsLoading(false);
      return;
    }

    try {
      await addDoc(collection(db, "comments"), {
        comment: desc,
        userId: auth?.currentUser ? auth.currentUser.uid : null,
        createdAt: moment().format(),
      });

      Alert.alert(
        t("write_comment.thanks_title"),
        t("write_comment.thanks_message"),
        [{ text: "OK", onPress: () => navigation.goBack() }]
      );
    } catch (e) {
      Alert.alert("Error sending comment:", `${e}`);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <SafeAreaView className="flex-1 bg-white">
      <KeyboardAwareScrollView
        className="p-4"
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        scrollEnabled
        behavior="padding"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
      >
        <View className="mt-5">
          <Text
            className="text-sm mb-2 ml-1"
            style={{ fontFamily: "Inter_400Regular" }}
          >
            {t("write_comment.message_placeholder")}
          </Text>
          <TextInput
            style={{ fontFamily: "Inter_500Medium" }}
            multiline
            textAlignVertical="top"
            className={TEXT_AREA_CLASS}
            placeholder=""
            placeholderTextColor={"gray"}
            value={desc}
            maxLength={MAX_DESC_LENGTH}
            onChangeText={(val) => {
              setDesc(val);
            }}
          />
          <Text
            className="text-xs text-gray-500 mt-2 ml-1 self-end"
            style={{ fontFamily: "Inter_300Light" }}
          >
            {MAX_DESC_LENGTH - desc.length} {t("write_comment.remaining_characters")}
          </Text>
        </View>
        <Text
          className="mt-10 text-xs leading-5 text-gray-500"
          style={{ fontFamily: "Inter_300Light" }}
        >
          {t("write_comment.thanks_message")}
        </Text>
        <View className="flex-1 mt-10 w-full">
          <PrimaryButton
            text={t("write_comment.button_text")}
            isLoading={isLoading}
            onPress={validate}
          />
        </View>
      </KeyboardAwareScrollView>
    </SafeAreaView>
  );
}
