import {
  View,
  Text,
  Linking,
  Alert,
  Pressable,
  Share,
  Platform,
  AppState,
  StatusBar,
  RefreshControl,
  ScrollView,
  ImageBackground,
  TouchableOpacity,
} from "react-native";
import React, {
  useCallback,
  useEffect,
  useLayoutEffect,
  useRef,
  useState,
} from "react";
import { addDoc, collection, doc, getDoc, updateDoc } from "firebase/firestore";
import { auth, db } from "../../../config/firebase";
import { sendEmailVerification, signOut } from "firebase/auth";
import moment from "moment/moment";
import PageSubTitle from "../../components/Titles/PageSubTitle";
import SettingsCard from "../../components/Settings/Card";
import PageLoader from "../../components/Loaders/PageLoader";
import * as Constants from "expo-constants";
import AsyncStorage from "@react-native-async-storage/async-storage";
import { COLORS } from "../../styles/colors";
import { useFocusEffect } from "@react-navigation/native";
import { Ionicons } from "@expo/vector-icons";

import Animated, { FadeIn, FadeOut } from "react-native-reanimated";
import { DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING } from "../../styles/constants";
import ProfileTopCard from "../../components/ProfileTopCard";
import BackButton from "../../components/Buttons/BackButton";

export default function Profile({ navigation }) {
  const scrollViewRef = useRef(null);
  const PADDING = "px-4 py-2";
  const buildNumber =
    Platform.OS === "ios"
      ? Constants.default.expoConfig.ios.buildNumber
      : Constants.default.expoConfig.android.versionCode;
  const [userInfo, setUserInfo] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [appState, setAppState] = React.useState(AppState.currentState);

  const appStoreURL = "https://apps.apple.com/us/app/id1641274230";

  const playStoreURL =
    "https://play.google.com/store/apps/details?id=com.sur4plot";
  const version = Constants.default.expoConfig.version;
  const website = "https://sur4plots.com/";

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: "Mon compte",
    });
  }, [navigation]);

  useLayoutEffect(
    useCallback(() => {
      Platform.OS === "ios"
        ? StatusBar.setBarStyle("dark-content")
        : (StatusBar.setBarStyle("light-content"),
          StatusBar.setBackgroundColor("#00000"));
    }, [])
  );

  useEffect(() => {
    getUserInfo();
  }, []);

  useFocusEffect(
    useCallback(() => {
      getUserInfo();
    }, [])
  );

  const topRefresh = () => {
    setIsLoading(true);
    auth?.currentUser?.reload();
    getUserInfo();
  };

  useEffect(() => {
    const subscription = AppState.addEventListener("change", (nextAppState) => {
      if (appState !== nextAppState) {
        setAppState(nextAppState);
        if (nextAppState == "active") {
          auth?.currentUser?.reload();
        }
      }
    });

    return () => {
      subscription.remove();
    };
  }, [appState]);

  const getUserInfo = async () => {
    if (!auth.currentUser) {
      setIsLoading(false);
      return;
    }
    try {
      const docRef = doc(db, "users", auth.currentUser.uid);

      const docSnap = await getDoc(docRef);

      if (docSnap.exists()) {
        setUserInfo({ id: docSnap.id, ...docSnap.data() });
      } else {
        Alert.alert(
          "Un problème est survenu",
          "Nous n'arrivons pas à récuperer vos données, veuillez contacter le support"
        );
        return;
      }
    } catch (error) {
      Alert.alert(
        "Un problème est survenu",
        `Nous n'arrivons pas à récuperer vos données, veuillez contacter le support: ${error}`
      );
      return;
    } finally {
      setIsLoading(false);
    }
  };

  const shareData = async () => {
    await Share.share({
      message: `"Découvrez Sur4Plots - la plateforme de référence pour la recherche et la récupération de véhicules volés ou perdus. Grâce à notre technologie avancée, retrouvez votre véhicule en un clin d'œil ! 🚗 Rejoignez notre communauté et participez à la lutte contre le vol automobile. Téléchargez dès maintenant notre application et bénéficiez de notre expertise. Cliquez ici pour commencer : \n${website}`,
      url: website,
    });
  };

  // const rateApp = async () => {
  //   // const urlIos = "https://apps.apple.com/us/app/m-services/id1625365961";
  //   const urlAndroid = "market://details?id=com.loka&showAllReviews=true";
  //   Linking.openURL(urlAndroid);
  //   // Linking.openURL(Platform.OS === "ios" ? urlIos : urlAndroid);
  // };

  const validateSignOut = () => {
    Alert.alert(
      "Se déconnecter",
      "Êtes vous sure de vouloir vous déconnecter?",
      [
        { text: "Déconnexion", onPress: () => logOut() },
        {
          text: "Annuler",
          style: "cancel",
        },
      ],
      {
        cancelable: true,
      }
    );
  };

  const logOut = async () => {
    setIsLoading(true);
    setTimeout(async () => {
      await AsyncStorage.removeItem("likedArticles");
      await signOut(auth);
      // navigation.popToTop();
    }, 500);
  };

  const onRefresh = async () => {
    setIsRefreshing(true);
    await getUserInfo();
    setIsRefreshing(false);
  };

  const validateUserDeleteAccount = () => {
    Alert.alert(
      "Demander la fermeture de mon compte",
      "Êtes vous sure?",
      [
        { text: "Fermer mon compte", onPress: () => userDeleteAccount() },
        {
          text: "Annuler",
          style: "cancel",
        },
      ],
      {
        cancelable: true,
      }
    );
  };

  const validateSendEmailValidation = () => {
    Alert.alert(
      `Vous n'avez pas reçu l'e-mail de vérification à ${auth.currentUser.email}`,
      "Nous allons vous renvoyer un e-mail contenant un lien de validation",
      [
        { text: "Renvoyer le lien", onPress: () => sendVerificationEmail() },
        {
          text: "Annuler",
          style: "cancel",
        },
      ],
      {
        cancelable: true,
      }
    );
  };

  function sendVerificationEmail() {
    sendEmailVerification(auth.currentUser)
      .then(() => {
        Alert.alert("L'e-mail de vérification vous a été envoyé");
      })
      .catch((error) => {
        if (error.code == "") {
          Alert.alert(
            "Une erreur est survenue lors de l'envoi de l'e-mail de vérification",
            "Vous avez fait trop de demandes, veuillez réessayer ultérieurement"
          );
          return;
        }
        Alert.alert(
          "Une erreur est survenue lors de l'envoi de l'e-mail de vérification",
          `${error}`
        );
      });
  }

  const becameHost = async () => {
    setIsLoading(true);
    await updateDoc(doc(db, "users", auth.currentUser?.uid), {
      "host.isHost": true,
    });
    getUserInfo();
  };

  const userDeleteAccount = async () => {
    setIsLoading(true);
    await addDoc(collection(db, "suppressAccountDemands"), {
      userId: auth.currentUser.uid,
      isCompleted: false,
      createdAt: moment().format(),
    })
      .then(() => {
        Alert.alert(
          "Demande envoyée avec succès",
          "Nous la traiterons dans les plus brefs délais (48h maximum)"
        );
      })
      .catch(() => {
        Alert.alert("Oops!", "Une erreur est survenue");
      });
    setIsLoading(false);
  };

  const Card = ({ number, title, onPress }) => (
    <Pressable
      onPress={onPress}
      className="w-[45%] p-5 mx-1 mb-4 rounded-xl bg-gray-100"
    >
      <Text
        className="text-2xl text-black text-left"
        style={{ fontFamily: "Inter_700Bold" }}
      >
        {number}
      </Text>
      <Text className="text-sm text-gray-500">{title}</Text>
    </Pressable>
  );

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <View className="bg-white">
      <ScrollView
        bounces={false}
        refreshControl={
          <RefreshControl
            refreshing={isRefreshing}
            onRefresh={onRefresh}
            tintColor={COLORS.primary}
          />
        }
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING / 2,
        }}
      >
        {/* first section */}
        <ProfileTopCard
          getUserInfo={getUserInfo}
          userInfo={auth.currentUser && userInfo}
        />

        {auth.currentUser && !auth?.currentUser?.emailVerified && (
          <Animated.View entering={FadeIn} exiting={FadeOut}>
            <Pressable
              onPress={validateSendEmailValidation}
              className="mx-4 border border-yellow-500 bg-yellow-50 rounded-md mb-3 p-2 flex-row items-center justify-between"
            >
              <View className="flex-row items-center gap-2">
               
                <Text
                  className="flex-1 text-sm text-yellow-700"
                  style={{ fontFamily: "Inter_300Light" }}
                >
                  Veuillez cliquer sur le lien envoyé à{" "}
                  <Text
                    className="flex-1 text-sm text-yellow-700"
                    style={{ fontFamily: "Inter_400Regular" }}
                  >
                    ({auth?.currentUser?.email})
                  </Text>{" "}
                  pour valider votre adresse e-mail {"\n"}
                  <Text
                    className="flex-1 text-sm text-yellow-700 underline"
                    style={{ fontFamily: "Inter_500Medium" }}
                  >
                    Je n'ai pas reçu de lien
                  </Text>
                </Text>
              </View>
            </Pressable>
          </Animated.View>
        )}

        {!isLoading &&
          auth?.currentUser &&
          !userInfo?.verification?.isVerified && (
            <View className={PADDING}>
              <PageSubTitle title={`Vérification`} />
              <SettingsCard
                green
                title={"Vérifier mes documents"}
                iconName={"checkmark-outline"}
                onPress={() =>
                  navigation.navigate("ValidateProfileStep1", { userInfo })
                }
              />
            </View>
          )}
        {/* second section */}

        {/* paramètre */}
        {auth?.currentUser && (
          <View className={PADDING}>
            <PageSubTitle title={"Compte"} />

            <SettingsCard
              title={"Informations personnelles"}
              iconName={"person-outline"}
              onPress={() =>
                navigation.navigate("PersonalInformations", {
                  data: userInfo,
                })
              }
            />

            <SettingsCard
              title={"Coordonnées bancaires"}
              iconName={"card-outline"}
              onPress={() =>
                navigation.navigate("SendRib", {
                  data: userInfo,
                })
              }
            />
          </View>
        )}
        {/* Traduction */}
        <View className={PADDING}>
          <PageSubTitle title={"Traduction"} />

          <SettingsCard
            title={"Changer de langue"}
            iconName={"language-outline"}
            onPress={shareData}
          />
        </View>

        {/* Commu */}
        <View className={PADDING}>
          <PageSubTitle title={"Communauté"} />

          <SettingsCard
            title={"Partager Sur4Plots"}
            iconName={"share-outline"}
            onPress={shareData}
          />

          <SettingsCard
            title={"Laisser un commentaire"}
            iconName={"mail-outline"}
            onPress={() => navigation.navigate("WriteComment")}
          />
        </View>

        {/* support */}
        <View className={PADDING}>
          <PageSubTitle title={"Support"} />

          <SettingsCard
            title={"Notre site web"}
            iconName={"globe-outline"}
            onPress={() => Linking.openURL(`https://www.sur4plots.com/`)}
          />

          <SettingsCard
            title={"Signaler un problème"}
            iconName={"bug-outline"}
            onPress={() => navigation.navigate("SignalBug")}
          />
        </View>
        {/* legal */}
        <View className={PADDING}>
          <PageSubTitle title={"Légal"} />

          <SettingsCard
            title={"Politique de confidentialité"}
            iconName={"newspaper-outline"}
            onPress={() =>
              Linking.openURL(
                "https://www.sur4plots.com/politique-de-confidentialite/"
              )
            }
          />
        </View>
        {/* Autres */}
        {auth.currentUser && (
          <View className={PADDING}>
            <PageSubTitle title={"Autres"} />

            <SettingsCard
              title={"Se déconnecter"}
              iconName={"log-out-outline"}
              onPress={validateSignOut}
            />
            <SettingsCard
              red
              title={"Supprimer mon compte"}
              iconName={"trash-outline"}
              onPress={validateUserDeleteAccount}
            />
          </View>
        )}
        <Text
          style={{ fontFamily: "Inter_400Regular" }}
          className="mt-10 text-center text-base text-gray-400"
        >
          Sur4Plots
        </Text>
        <Text
          style={{ fontFamily: "Inter_400Regular" }}
          className="text-center text-base text-gray-400"
        >
          {Constants.default.expoConfig.version} ({buildNumber})
        </Text>
      </ScrollView>
    </View>
  );
}
