import { View, Text } from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import { auth, db } from "../../../config/firebase";
import {
  addDoc,
  collection,
  doc,
  getDoc,
  getDocs,
  limit,
  onSnapshot,
  orderBy,
  query,
  startAfter,
  updateDoc,
  where,
} from "firebase/firestore";
import { Pressable } from "react-native";
import { SafeAreaView } from "react-native";
import {
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  LISTING_LIMIT,
} from "../../styles/constants";
import PageLoader from "../../components/Loaders/PageLoader";
import { FlashList } from "@shopify/flash-list";
import ListIcons from "../../components/ListIcons";
import moment from "moment";

export default function FindedSearchs({ navigation }) {
  const [data, setData] = useState([]);
  const [isLoading, setIsLoading] = useState(true);

  const [lastVisible, setLastVisible] = useState(null);
  const [hasMore, setHasMore] = useState(true);

  useEffect(() => {
    getData();
  }, []);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerTitle: "Matchs",
    });
  }, [navigation]);

  async function getData(lastDoc) {
    try {
      let q = query(
        collection(db, "searchs"),
        where("status", "==", "ended"),
        orderBy("findedAt", "desc"),
        limit(LISTING_LIMIT)
      );
      if (lastDoc) {
        q = query(q, startAfter(lastDoc));
      }
      const querySnapshot = await getDocs(q);
      const dataArr = [];
      let lastVisibleDoc = null;

      querySnapshot.forEach(async (res) => {
        dataArr.push({
          id: res.id,
          ...res.data(),
        });
        lastVisibleDoc = res;
      });
      if (lastDoc) {
        setData((prevData) => [...prevData, ...dataArr]);
      } else {
        setData(dataArr);
      }

      setLastVisible(lastVisibleDoc);
      setHasMore(dataArr.length === LISTING_LIMIT);
    } catch (error) {
      console.log(error);
    } finally {
      setIsLoading(false);
    }
  }

  const renderFooter = () => {
    if (!isLoading) return null;
    return (
      <View className="pt-5">
        <Loader />
      </View>
    );
  };

  const handleLoadMore = () => {
    if (!isLoading) {
      getData(lastVisible);
    }
  };

  const updateIsActive = async (value, itemId) => {
    const docRef = doc(db, "searchs", itemId);
    await updateDoc(docRef, {
      isActive: !value,
    });
    getData();
  };

  const Card = ({ item }) => (
    <Pressable
      className="p-3 bg-gray-50 border border-gray-200 mt-2 rounded-md flex-row items-center justify-between"
      onPress={() => navigation.navigate("SearchDetails", { item })}
    >
      <View className="flex-1 flex-row items-center">
        <View>
        <View className="mt-1 flex-row items-center">
            <Text
              className="text-base"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Plaque d'immatriculation: {item?.plate.toUpperCase()}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-base text-blue-600"
              style={{ fontFamily: "Inter_500Medium" }}
            >
              {item.status == "ended" ? `Retrouvé` : "En recherche"}
              {item.status == "ended" &&
                item.paymentStatus == "paid" &&
                " · Payé"}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-base"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              {item.status == "ended"
                ? `Position: ${
                    item?.address.street +
                    " · " +
                    item?.address.city +
                    " " +
                    item?.address.postalCode +
                    ", " +
                    item?.address.country
                  }`
                : "En recherche"}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-sm text-gray-500"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Retrouvé le{" "}
              {moment(item?.findedAt).format("DD/MM/YYYY à hh:mm a")}
            </Text>
          </View>
        </View>
      </View>
      {/* <View className="">
        <ListIcons
          two
          onPress2={() => navigation.navigate("EditCategory", { item })}
          three
          onPress3={() =>
            updateIsActive(
              item.isActive == undefined ? true : item.isActive ? true : false,
              item.id
            )
          }
          itemIsActive={
            item.isActive == undefined ? true : item.isActive ? true : false
          }
        />
      </View> */}
    </Pressable>
  );

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-gray-100">
      <FlashList
        estimatedItemSize={50}
        className="p-4"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
        keyExtractor={(item) => item.id}
        data={data}
        renderItem={({ item }) => <Card item={item} />}
        ListFooterComponent={renderFooter}
        onEndReached={handleLoadMore}
        onEndReachedThreshold={0.1}
      />
    </SafeAreaView>
  );
}
