import {
  View,
  Text,
  SafeAreaView,
  Image,
  TouchableOpacity,
} from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import { DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING } from "../../styles/constants";
import moment from "moment";
import { auth, db } from "../../../config/firebase";
import PageLoader from "../../components/Loaders/PageLoader";
import { doc, getDoc } from "firebase/firestore";
import PageSubTitle from "../../components/Titles/PageSubTitle";
import PageTitle from "../../components/Titles/PageTitle";
import { COLORS } from "../../styles/colors";
import { formatNumber } from "../../utils/allFunctions";
import ImageViewer from "react-native-image-zoom-viewer";
import Modal from "react-native-modal";
import { XMarkIcon } from "react-native-heroicons/solid";

export default function SearchDetails({ navigation, route }) {
  const { item } = route.params;
  const [finderInfo, setFinderInfo] = useState(null);

  const [imageForViewer, setImageForViewer] = useState(null);
  const [isPhotoModalOpen, setIsPhotoModalOpen] = useState(false);

  const [isLoading, setIsLoading] = useState(true);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerTitle: "Détails",
    });
  }, [navigation]);

  useEffect(() => {
    getUserInfo();
  }, []);

  const getUserInfo = async () => {
    if (!item?.status == "ended") {
      setIsLoading(false);
      return;
    }
    try {
      const imageUrls = item.photos.map((image) => ({ url: image }));
      setImageForViewer(imageUrls);

      const docRef = doc(db, "users", item.finderId);
      const docSnap = await getDoc(docRef);

      if (docSnap.exists()) {
        setFinderInfo({ id: docSnap.id, ...docSnap.data() });
      } else {
        Alert.alert(
          "Un problème est survenu",
          "Nous n'arrivons pas à récuperer vos données, veuillez contacter le support"
        );
        return;
      }
    } catch (error) {
      Alert.alert(
        "Un problème est survenu",
        `Nous n'arrivons pas à récuperer vos données, veuillez contacter le support: ${error}`
      );
      return;
    } finally {
      setIsLoading(false);
    }
  };

  const CardList = ({ title, value }) => (
    <View className="py-2">
      <Text
        className="text-gray-500 text-sm"
        style={{ fontFamily: "Inter_300Light" }}
      >
        {title}
      </Text>
      <Text
        className="uppercase text-base"
        style={{ fontFamily: "Inter_400Regular" }}
      >
        {value}
      </Text>
    </View>
  );

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-gray-100">
      <KeyboardAwareScrollView
        className="p-2"
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        // resetScrollToCoords={{ x: 0, y: 0 }}
        scrollEnabled
        behavior="padding"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
      >
        <View className='rounded-md p-2 bg-white'>
          <PageTitle
            color={COLORS.primary}
            title={"Informations du véhicule"}
          />
          <CardList title={"Plaque d'immatriculation"} value={item?.plate} />
          <CardList title={"Assurance"} value={item?.assurance} />
          <CardList title={"Type de véhicule"} value={item?.type} />
          <CardList
            title={"Publié le"}
            value={moment(item?.createdAt).format("DD/MM/YYYY à hh:mm a")}
          />
          <CardList
            title={"État"}
            value={item?.status == "ended" ? "Retrouvé" : "En recherche"}
          />
          {item?.status == "ended" && (
            <CardList
              title={"Le client a payé le"}
              value={moment(item?.paiedAt).format("DD/MM/YYYY à hh:mm a")}
            />
          )}
        </View>

        {item?.status == "ended" && item.finderId && (
          <View className='mt-5 rounded-md p-2 bg-white'>
            <PageTitle
              color={COLORS.primary}
              title={"Informations du trouveur"}
            />
            <CardList title={"Email"} value={finderInfo?.email} />
            <CardList title={"Nom"} value={finderInfo?.lastName} />
            <CardList title={"Prénom"} value={finderInfo?.firstName} />
            <CardList
              title={"A connu Sur4Plots par"}
              value={!finderInfo?.knowing ? "Pas rempli" : finderInfo?.knowing}
            />
            <CardList
              title={"Compte vérifié"}
              value={finderInfo?.verification?.isVerified ? "Oui" : "Non"}
            />
            <CardList title={"ID"} value={finderInfo?.id} />
            <CardList
              title={"Retrouvé le"}
              value={moment(item?.findedAt).format("DD/MM/YYYY à hh:mm a")}
            />
            <CardList
              title={"A été crédité"}
              value={
                item?.finderPaymentStatus == "paied"
                  ? "a été payé"
                  : "pas encore"
              }
            />
            <View className='mt-5'>
              <PageTitle color={COLORS.primary} title={"Photos du véhicule"} />
            </View>
            <TouchableOpacity
              activeOpacity={0.7}
              onPress={() => setIsPhotoModalOpen(true)}
              className="flex-row items-center justify-between"
            >
              {item.photos.map((item) => (
                <Image
                  className="w-[48%] h-[15vh] rounded-md"
                  source={{ uri: item }}
                />
              ))}
            </TouchableOpacity>
          </View>
        )}

        {item?.status == "ended" && item.finderId && (
          <View >
            <PageTitle
              color={COLORS.primary}
              title={"Coordonnées bancaires du trouveur"}
            />
            <CardList
              title={"IBAN"}
              value={
                finderInfo?.bankInfo?.iban
                  ? finderInfo?.bankInfo?.iban
                  : "Pas rempli"
              }
            />
            <CardList
              title={"BIC/SWIFT"}
              value={
                finderInfo?.bankInfo?.bic
                  ? finderInfo?.bankInfo?.bic
                  : "Pas rempli"
              }
            />
            <CardList
              title={"Nom de la banque"}
              value={
                finderInfo?.bankInfo?.bankName
                  ? finderInfo?.bankInfo?.bankName
                  : "Pas rempli"
              }
            />
          </View>
        )}
      </KeyboardAwareScrollView>
      <Modal
        style={{
          flex: 1,
          position: "absolute",
          width: "100%",
          height: "100%",
          padding: 0,
          margin: 0,
        }}
        visible={isPhotoModalOpen}
        transparent={true}
      >
        <ImageViewer
          renderHeader={() => (
            <>
              <View
                style={{
                  top: 50,
                  zIndex: 200,
                }}
                className="absolute w-full"
              >
                <TouchableOpacity
                  className="p-5"
                  style={{ alignSelf: "flex-end", marginRight: 10 }}
                  onPress={() => {
                    setIsPhotoModalOpen(false);
                  }}
                >
                  <XMarkIcon size={30} color={"white"} />
                </TouchableOpacity>
              </View>
              <View
                style={{
                  bottom: 50,
                  zIndex: 200,
                }}
                className="absolute w-full"
              >
                <Text
                  style={{ fontFamily: "Inter_700Bold" }}
                  className="text-white text-xl text-center"
                >
                  {item.plate}
                </Text>
              </View>
            </>
          )}
          enableSwipeDown
          onCancel={() => setIsPhotoModalOpen(false)}
          onSwipeDown={() => setIsPhotoModalOpen(false)}
          enableImageZoom
          imageUrls={imageForViewer}
        />
      </Modal>
    </SafeAreaView>
  );
}
