import { View, Text } from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import { auth, db } from "../../../config/firebase";
import {
  collection,
  doc,
  getDoc,
  getDocs,
  orderBy,
  query,
  where,
} from "firebase/firestore";
import { Pressable } from "react-native";
import { SafeAreaView } from "react-native";
import moment from "moment";
import { DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING } from "../../styles/constants";
import PageLoader from "../../components/Loaders/PageLoader";
import { FlashList } from "@shopify/flash-list";

export default function Finders({ navigation }) {
  const [data, setData] = useState([]);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    fetchFindersData();
  }, []);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerTitle: "Trouveurs",
    });
  }, [navigation]);

  const fetchFindersData = async () => {
    try {
      let q = query(
        collection(db, "searchs"),
        where("finderId", "!=", null),
        orderBy("createdAt", "desc")
      );

      const searchQuerySnapshot = await getDocs(q);
      const finderPromises = [];

      searchQuerySnapshot.forEach((searchDoc) => {
        const searchData = searchDoc.data();
        if (searchData.finderId) {
          const userDocRef = doc(db, "users", searchData.finderId);
          finderPromises.push(
            getDoc(userDocRef).then((userDoc) => {
              return {
                searchId: searchDoc.id,
                ...searchData,
                finderInfo: userDoc.exists()
                  ? { id: userDoc.id, ...userDoc.data() }
                  : null,
              };
            })
          );
        }
      });

      const combinedData = await Promise.all(finderPromises);
      setData(combinedData);
    } catch (error) {
      console.error("Error fetching finders data: ", error);
    } finally {
      setIsLoading(false);
    }
  };

  const getClassName = (finderPaymentStatus) => {
    if (finderPaymentStatus === "unpaid") {
      return "bg-red-200";
    }
    if (finderPaymentStatus === "paid") {
      return "bg-green-200";
    }
    return "bg-white";
  };

  const Card = ({ item }) => (
    <Pressable
      className={`p-3 ${getClassName(
        item.finderPaymentStatus
      )} border border-gray-200 mt-2 rounded-md flex-row items-center justify-between`}
      onPress={() =>
        navigation.navigate("FinderDetails", { item: item })
      }
    >
      <View className="flex-1 flex-row items-center">
        <View>
          <View className="flex-row items-center">
            <Text
              className="text-base text-gray-700"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Nom: {item?.finderInfo?.firstName + " "}
              <Text
                className="text-base text-gray-700"
                style={{ fontFamily: "Inter_500Medium" }}
              >
                {item?.finderInfo?.lastName}
              </Text>
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-base"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Plaque d'immatriculation: {item?.plate.toUpperCase()}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-base"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Nom de l'assurance: {item?.assurance}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-base"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Récompense: {item.finderPaymentStatus == "paid"
                ? "A été Payé"
                : "N'a pas été payé"}
            </Text>
          </View>
          <View className="mt-1 flex-row items-center">
            <Text
              className="text-sm text-blue-600"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              Véhicule aperçu {moment(item?.findedAt).format("DD/MM/YYYY à hh:mm a")}
            </Text>
          </View>
        </View>
      </View>
    </Pressable>
  );

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-gray-100">
      <FlashList
        estimatedItemSize={50}
        className="p-4"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
        keyExtractor={(item, index) => index.toString()}
        data={data}
        renderItem={({ item }) => <Card item={item} />}
      />
    </SafeAreaView>
  );
}
