import {
  View,
  Text,
  SafeAreaView,
  ScrollView,
  TouchableOpacity,
  Pressable,
  Platform,
  StatusBar,
  Keyboard,
  Alert,
  ImageBackground,
} from "react-native";
import React, {
  useCallback,
  useEffect,
  useLayoutEffect,
  useMemo,
  useRef,
  useState,
} from "react";

import { Ionicons } from "@expo/vector-icons";
import { COLORS } from "../styles/colors";
import {
  BottomSheetBackdrop,
  BottomSheetFlatList,
  BottomSheetModal,
  BottomSheetModalProvider,
  BottomSheetScrollView,
  BottomSheetTextInput,
} from "@gorhom/bottom-sheet";
import {
  collection,
  doc,
  getDoc,
  getDocs,
  onSnapshot,
  query,
  setDoc,
  updateDoc,
  where,
} from "firebase/firestore";
import { auth, db } from "../../config/firebase";
import { Image } from "react-native";
import Loader from "../components/Loaders/Loader";
import Modal from "react-native-modal";
import {
  BUTTON_CLASS,
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  INPUT_CLASS_BORDER_BOTTOM,
  INPUT_CLASS_BORDER_BOTTOM_BG_WHITE,
} from "../styles/constants";
import * as Device from "expo-device";
import * as Notifications from "expo-notifications";
import Constants from "expo-constants";
import PageSubTitle from "../components/Titles/PageSubTitle";
import PageTitle from "../components/Titles/PageTitle";
import PrimaryButton from "../components/Buttons/PrimaryButton";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import { FlashList } from "@shopify/flash-list";
import { COUNTRIES } from "../utils/countries";
import { formatNumber, openSettings } from "../utils/allFunctions";
import { TIPS, TIPS_EN } from "../utils/tips";
import SmallPrimaryButton from "../components/Buttons/SmallPrimaryButton";
import { useFocusEffect } from "@react-navigation/native";
import Animated, {
  BounceIn,
  BounceOut,
  FadeIn,
  FadeInLeft,
  FadeInRight,
  FadeOut,
  FadeOutRight,
  SlideInDown,
} from "react-native-reanimated";
import PageLoader from "../components/Loaders/PageLoader";
import { useTranslation } from "react-i18next";
import frFlag from "../../assets/img/fr.png";
import enFlag from "../../assets/img/en.png";
import { TextInput } from "react-native-gesture-handler";
import { Badge } from "react-native-elements";

Notifications.setNotificationHandler({
  handleNotification: async () => ({
    shouldShowAlert: true,
    shouldPlaySound: false,
    shouldSetBadge: false,
  }),
});

export default function Home({ navigation }) {
  const { t, i18n } = useTranslation();
  const notificationListener = useRef();
  const responseListener = useRef();

  const [selectedCountry, setSelectedCountry] = useState(COUNTRIES[0]);
  const [vehicleId, setVehicleId] = useState("");
  const [findedVehicleId, setFindedVehicleId] = useState("");
  const [assurance, setAssurance] = useState("");

  const [userInfo, setUserInfo] = useState(null);

  const [cgvData, setCgvData] = useState();

  const [notifNumber, setNotifNumber] = useState(0);

  const [isLoading, setIsLoading] = useState(true);

  const [maj, setMaj] = useState(false);
  const [notif, setNotif] = useState(false);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerTitleStyle: {
        color: "white",
        fontSize: 20,
      },
      headerStyle: {
        backgroundColor: COLORS.primary,
      },
      headerLeft: () => (
        <View className="flex-row items-center bg-white rounded-full px-2">
          <Text
            style={{ fontFamily: "Inter_500Medium", color: COLORS.primary }}
            className="text-base"
          >
            {t("text.homeConnected")}
          </Text>
          <View className="ml-2 flex-row items-center  rounded-full">
            {/* <Ionicons name="radio-button-on" size={18} color="green" /> */}
            <Badge status="success" />
          </View>
        </View>
      ),
      headerTitle: "",
      headerRight: () => (
        <View className="flex-row items-center">
          <TouchableOpacity
            onPress={openModal}
            className="flex-row items-center"
          >
            <Text
              style={{ fontFamily: "Inter_500Medium" }}
              className="text-xl text-white"
            >
              {i18n?.language == "en" ? "EN" : "FR"}
            </Text>
            <Image
              source={i18n?.language == "en" ? enFlag : frFlag}
              className="mx-1 w-6 h-4 z-20"
            />
            <Ionicons name={`chevron-down-outline`} size={15} color={"white"} />
          </TouchableOpacity>
        </View>
      ),
    });
  });

  useFocusEffect(
    useCallback(() => {
      Platform.OS === "ios"
        ? StatusBar.setBarStyle("light-content")
        : StatusBar.setBarStyle("light-content");
      return () => {
        Platform.OS === "ios"
          ? StatusBar.setBarStyle("dark-content")
          : StatusBar.setBarStyle("dark-content");
      };
    }, [])
  );

  useEffect(() => {
    checkMaj();
    if (!auth.currentUser) {
      setIsLoading(false);
      return;
    }
    getNotifs();
    getUserInfo();
    getCGV();
    getToken();
  }, []);

  useEffect(() => {
    notificationListener.current =
      Notifications.addNotificationReceivedListener((notification) => {});

    responseListener.current =
      Notifications.addNotificationResponseReceivedListener((response) => {
        navigation.jumpTo("DemandsTab");
      });

    return () => {
      Notifications.removeNotificationSubscription(
        notificationListener.current
      );
      Notifications.removeNotificationSubscription(responseListener.current);
    };
  }, []);

  async function getCGV() {
    const docRef = doc(db, "CGV", "CGV");
    const docSnap = await getDoc(docRef);

    if (docSnap.exists()) {
      setCgvData(docSnap.data());
    }
  }

  const getNotifs = async () => {
    if (!auth.currentUser) {
      setIsLoading(false);
      return;
    }
    try {
      const querySnapshot = await getDocs(
        query(
          collection(db, "notifications"),
          where("userId", "==", auth?.currentUser.uid),
          where("isViewed", "==", false)
        )
      );
      setNotifNumber(querySnapshot.size);
      await Notifications.setBadgeCountAsync(querySnapshot.size);
    } catch (error) {
      console.error("Error fetching data:", error.message);
    } finally {
      setIsLoading(false);
    }
  };

  const getUserInfo = async () => {
    try {
      const docRef = doc(db, "users", auth?.currentUser.uid);
      const docSnap = await getDoc(docRef);

      if (docSnap.exists()) {
        setUserInfo(docSnap.data());
      }
    } catch (error) {
      console.log(error);
    } finally {
      setIsLoading(false);
    }
  };

  const checkMaj = async () => {
    const appStoreURL = "";
    const playStoreURL =
      "https://play.google.com/store/apps/details?id=com.sur4plots";
    const version = Constants.default.expoConfig.version;

    const check = await checkVersion({
      version: version,
      // iosStoreURL: appStoreURL,
      androidStoreURL: playStoreURL,
      country: "fr",
    });
    if (check.result === "new") {
      setMaj(true);
    } else {
      setMaj(false);
    }
  };

  async function registerForPushNotificationsAsync() {
    let token;
    if (Device.isDevice) {
      const { status: existingStatus } =
        await Notifications.getPermissionsAsync();
      let finalStatus = existingStatus;
      if (existingStatus !== "granted") {
        const { status } = await Notifications.requestPermissionsAsync();
        finalStatus = status;
      }
      if (finalStatus !== "granted") {
        setNotif(true);
        return;
      }
      let token = await Notifications.getExpoPushTokenAsync({
        projectId: Constants.expoConfig.extra.eas.projectId,
      });
      setNotif(false);
      return token.data;
    } else {
      // alert("Must use physical device for Push Notifications");
    }

    if (Platform.OS === "android") {
      Notifications.setNotificationChannelAsync("default", {
        name: "default",
        importance: Notifications.AndroidImportance.MAX,
        vibrationPattern: [0, 250, 250, 250],
        lightColor: "#FF231F7C",
      });
    }

    return token;
  }

  async function getToken() {
    registerForPushNotificationsAsync().then((token) =>
      addExpoPushTokenToUser(token)
    );
  }

  async function addExpoPushTokenToUser(token) {
    await setDoc(
      doc(db, "users", auth.currentUser.uid),
      {
        expoPushToken: token == "" ? "" : token,
      },
      { merge: true }
    );
  }

  const bottomSheetModalRef = useRef(null);

  // variables
  const snapPoints = useMemo(() => ["40%", "70%"], []);

  // callbacks
  const openModal = useCallback(() => {
    Keyboard?.dismiss();
    bottomSheetModalRef.current?.present();
  }, []);

  const closeModal = useCallback(() => {
    bottomSheetModalRef.current?.dismiss();
  }, []);

  const renderBackdrop = useCallback(
    (props) => (
      <BottomSheetBackdrop
        {...props}
        disappearsOnIndex={-1}
        // appearsOnIndex={1}
      />
    ),
    []
  );

  const Card = ({ title, desc, img, link, jumpTo }) => (
    <TouchableOpacity
      onPress={() => {
        if (
          auth.currentUser &&
          !auth?.currentUser?.emailVerified &&
          (link == "SearchStep1" || link == "Find")
        ) {
          alert(t("screens.profile.account.emailinfo"));
          navigation.navigate(`Profile`);
          return;
        }
        jumpTo
          ? navigation.jumpTo(`${link}`)
          : navigation.navigate(`${link}`, { userInfo });
      }}
      className="p-3 mb-3 border border-gray-200 rounded-md flex flex-row items-center"
    >
      <Image
        style={{
          height: 60,
          width: 60,
        }}
        resizeMode="contain"
        source={img}
      />

      <View className="ml-3 flex-1">
        <Text style={{ fontFamily: "Inter_500Medium" }} className="text-xl">
          {title}
        </Text>
        <Text
          style={{ fontFamily: "Inter_400Regular" }}
          className="text-sm text-gray-500 flex"
        >
          {desc}
        </Text>
      </View>
    </TouchableOpacity>
  );

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-white">
      <KeyboardAwareScrollView
        className="p-4"
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        scrollEnabled
        behavior="padding"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
      >
        <View className="flex">
          <Card
            title={t("mon_profil.title")}
            desc={t("mon_profil.desc")}
            img={require("../../assets/img/femme.png")}
            link={"Profile"}
          />
          <Card
            title={t("je_cherche_un_vehicule.title")}
            desc={t("je_cherche_un_vehicule.desc")}
            img={require("../../assets/img/plus.png")}
            link={"SearchStep1"}
          />
          <Card
            title={t("j_ai_trouve_un_vehicule.title")}
            desc={t("j_ai_trouve_un_vehicule.desc")}
            img={require("../../assets/img/find.png")}
            link={"Find"}
          />
          <Card
            title={t("mes_demarches.title")}
            desc={t("mes_demarches.desc")}
            img={require("../../assets/img/search.png")}
            jumpTo={true}
            link={"DemandsTab"}
          />
          <Card
            title={t("notifications.title")}
            desc={t("notifications.desc")}
            img={require("../../assets/img/notification.png")}
            jumpTo={true}
            link={"NotifTab"}
          />

          <Card
            title={t("comment_ca_marche.title")}
            desc={t("comment_ca_marche.desc")}
            img={require("../../assets/img/info.png")}
            link={"HowItsWork"}
          />
        </View>
      </KeyboardAwareScrollView>
      {/* no auth user  */}
      {!auth?.currentUser && (
        <Animated.View entering={SlideInDown} exiting={FadeOut}>
          <TouchableOpacity
            activeOpacity={0.7}
            onPress={() => navigation.navigate("Landing")}
            style={{ backgroundColor: "rgba(46, 204, 113, 0.9)" }}
            className="w-full p-2 self-center absolute w-11/12 rounded-md bottom-2
             z-20 flex-row items-center justify-between"
          >
            <View className="flex-row items-center w-6/12">
              {/* Text de mise à jour */}
              <Text
                className="ml-2 text-xs text-white"
                style={{ fontFamily: "Inter_400Regular" }}
              >
                {t("noAuth.text")}
              </Text>
            </View>
            <TouchableOpacity
              activeOpacity={0.7}
              onPress={() => navigation.navigate("Landing")}
              className="rounded-full bg-white px-2 py-1 flex-row items-center justify-center"
            >
              <Text
                className="text-xs text-green-500 text-center"
                style={{ fontFamily: "Inter_400Regular" }}
              >
                {t("noAuth.btn")}
              </Text>
            </TouchableOpacity>
          </TouchableOpacity>
        </Animated.View>
      )}

      {/* new mise à jour */}
      {notif && (
        <Animated.View
          entering={SlideInDown}
          exiting={FadeOut}
          className="p-2 self-center absolute w-11/12 rounded-md bottom-5 bg-blue-500 z-20 flex-row items-center justify-between"
        >
          <View className="flex-row items-center w-6/12">
            {/* plus tard */}
            <TouchableOpacity onPress={() => setNotif(false)} className="p-1">
              <Ionicons
                style={{ alignSelf: "center" }}
                name={`close-outline`}
                size={25}
                color={"white"}
              />
            </TouchableOpacity>
            {/* Text de mise à jour */}
            <Text
              className="ml-2 text-xs text-white"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              {t("notif.text")}
            </Text>
          </View>
          <TouchableOpacity
            onPress={openSettings}
            activeOpacity={0.9}
            className="rounded-full bg-white px-2 py-1 flex-row items-center justify-center"
          >
            <Text
              className="text-xs text-blue-500 text-center"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              {t("notif.btn")}
            </Text>
          </TouchableOpacity>
        </Animated.View>
      )}
      <BottomSheetModalProvider>
        <BottomSheetModal
          ref={bottomSheetModalRef}
          backdropComponent={renderBackdrop}
          index={0}
          snapPoints={snapPoints}
        >
          <View className="p-2">
            <TouchableOpacity
              className="flex-row items-center pb-3"
              onPress={() => {
                i18n.changeLanguage("fr");
                setIsLoading(true);
                setTimeout(() => {
                  setIsLoading(false);
                  closeModal();
                }, 2000);
              }}
            >
              <Image
                source={require("../../assets/img/fr.png")}
                className="mr-4 rounded-full w-10 h-10 z-20"
              />
              <Text
                style={{ fontFamily: "Inter_500Medium" }}
                className="text-lg mr-2"
              >
                {t("text.french")}
              </Text>
              {i18n?.language == "fr" && <Badge status="success" />}
            </TouchableOpacity>
            <TouchableOpacity
              className="flex-row items-center py-3 border-y border-gray-200"
              onPress={() => {
                i18n.changeLanguage("en");
                setIsLoading(true);
                setTimeout(() => {
                  setIsLoading(false);
                  closeModal();
                }, 2000);
              }}
            >
              <Image
                source={require("../../assets/img/en.png")}
                className="mr-4 rounded-full w-10 h-10 z-20"
              />
              <Text
                style={{ fontFamily: "Inter_500Medium" }}
                className="text-lg mr-2"
              >
                {t("text.english")}
              </Text>
              {i18n?.language == "en" && <Badge status="success" />}
            </TouchableOpacity>
          </View>
        </BottomSheetModal>
      </BottomSheetModalProvider>
    </SafeAreaView>
  );
}
