import {
  View,
  Text,
  TouchableOpacity,
  Image,
  Alert,
  StatusBar,
} from "react-native";
import React, {
  useCallback,
  useEffect,
  useLayoutEffect,
  useState,
} from "react";
import {
  collection,
  doc,
  getDoc,
  getDocs,
  orderBy,
  query,
  updateDoc,
  where,
} from "firebase/firestore";
import { auth, db } from "../../config/firebase";
import { SafeAreaView } from "react-native";
import { FlatList } from "react-native";
import moment from "moment";
import PageLoader from "../components/Loaders/PageLoader";
import { useFocusEffect } from "@react-navigation/native";
import { useTranslation } from "react-i18next";

export default function Notification({ navigation }) {
  const [data, setData] = useState([]);
  const [isLoading, setIsLoading] = useState(true);
  const { t, i18n } = useTranslation();



  useLayoutEffect(() => {
    navigation.setOptions({
      headerTitle: "Notifications",
    });
  });

  useEffect(() => {
    if (!auth.currentUser) {
      setIsLoading(false);
      return;
    }
    getNotifs();
  }, []);

  useFocusEffect(
    useCallback(() => {
      StatusBar.setBarStyle("dark-content");
    }, [])
  );

  async function getNotifs() {
    try {
      const q = query(
        collection(db, "notifications"),
        where("userId", "==", auth.currentUser.uid),
        orderBy("createdAt", "desc")
      );
      const querySnapshot = await getDocs(q);
      const dataArr = [];
      querySnapshot.forEach(async (res) => {
        dataArr.push({
          id: res.id,
          ...res.data()
        });
      });
      setData(dataArr);
    } catch (error) {
      console.log(`${error}`);
    } finally {
      setIsLoading(false);
    }
  }

  async function updateData(itemId, isViewed) {
    if (isViewed) {
      return;
    }
    const docRef = doc(db, "notifications", itemId);
    await updateDoc(docRef, {
      isViewed: true,
    });
    getNotifs();
  }

  if (isLoading) {
    return <PageLoader />;
  }

  const Card = ({ item }) => (
    <TouchableOpacity
      activeOpacity={0.7}
      onPress={() => {
        updateData(item?.id, item?.isViewed);
        navigation.jumpTo("DemandsTab");
      }}
      className="w-full px-4 py-2 mb-2 rounded-md bg-white border border-gray-100"
    >
      <Text
        className="text-lg text-black mb-2"
        style={{ fontFamily: "Inter_500Medium" }}
      >
        {i18n?.language === "en" && item?.title_en
          ? item?.title_en
          : item?.title}
      </Text>
      <View className="flex-row items-center gap-3 w-full">
        {!item?.isViewed && <View className="w-3 h-3 bg-red-500 rounded-full"></View>}
        <Text className="text-base text-gray-700">
          {i18n?.language === "en" && item?.text_en ? item?.text_en : item?.text}
        </Text>
      </View>
      <Text className="text-sm mt-4 self-end text-gray-400">
        {moment(item?.createdAt).format("DD/MM/YYYY à h:mm a")}
      </Text>
    </TouchableOpacity>
  );

  if (data.length < 1) {
    return (
      <View className="flex-1 items-center justify-center px-[10%] -mt-[30%]">
        <Image
          source={require("../../assets/img/notification.png")}
          resizeMode="contain"
          className="w-24 h-24"
        />
        <Text
          className="text-base text-center text-black mt-10 mb-2"
          style={{ fontFamily: "Inter_500Medium" }}
        >
          {t("notifications.noNotifTitle")}
        </Text>
        <Text className="text-sm text-center text-gray-700">
          {t("notifications.noNotifDesc")}
        </Text>
      </View>
    );
  }

  return (
    <SafeAreaView className="flex-1 bg-gray-100">
      <FlatList
        className="p-3"
        contentContainerStyle={{
          paddingBottom: "50%",
        }}
        keyExtractor={(item) => item.id}
        data={data}
        renderItem={({ item }) => <Card item={item} />}
      />
    </SafeAreaView>
  );
}
