import {
  Text,
  SafeAreaView,
  TouchableOpacity,
  Alert,
  Image,
} from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import { collection, getDocs, query } from "firebase/firestore";
import { db } from "../../../config/firebase";
import { DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING } from "../../styles/constants";
import { FlashList } from "@shopify/flash-list";
import PageLoader from "../../components/Loaders/PageLoader";
import { useTranslation } from "react-i18next";
import BackButton from "../../components/Buttons/BackButton";

const SearchStep2 = ({ navigation, route }) => {
  const { selectedCountry, userInfo } = route.params;
  const { t, i18n } = useTranslation();
  const [cat, setCat] = useState(null);
  const [cats, setCats] = useState([]);
  const [isLoading, setIsLoading] = useState(true);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: t("vehicleType.headerTitle"),
    });
  }, [navigation, t]);

  useEffect(() => {
    getCategories();
  }, []);

  async function getCategories() {
    const q = query(collection(db, "categories"));
    const querySnapshot = await getDocs(q);
    const dataArr = [];
    querySnapshot.forEach((res) => {
      dataArr.push({
        id: res.id,
        ...res.data(),
      });
    });

    dataArr.sort(customSort);
    setCats(dataArr);
    setIsLoading(false);
  }

  const customSort = (a, b) => {
    const nameA = a.name.toLowerCase();
    const nameB = b.name.toLowerCase();
    

    // Mettre "voiture" en premier
    if (nameA.startsWith("voiture") && !nameB.startsWith("voiture")) {
      return -1;
    }
    if (!nameA.startsWith("voiture") && nameB.startsWith("voiture")) {
      return 1;
    }

    // Trier par ordre alphabétique
    return nameA.localeCompare(nameB);
  };

  async function validate(item) {
    if (!item) {
      Alert.alert(t("vehicleType.chooseType"));
      return;
    }
    navigation.navigate("SearchStep3", {
      selectedCountry,
      cat: item,
      userInfo,
    });
  }

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-white">
      <FlashList
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING / 2,
        }}
        data={cats}
        renderItem={({ item }) => (
          <TouchableOpacity
            className="flex-row items-center p-2 border-b border-gray-200"
            onPress={() => {
              validate(item.name.toLowerCase());
            }}
          >
            <Image
              className="mr-3 w-14 h-14 z-20 rounded-full"
              source={{ uri: item.image }}
              resizeMode={"cover"}
            />
            <Text
              style={{ fontFamily: "Inter_400Regular" }}
              className="text-base"
            >
              {i18n?.language === "en"
                ? item.name_en.charAt(0).toUpperCase() + item.name_en.slice(1)
                : item.name.charAt(0).toUpperCase() + item.name.slice(1)}
            </Text>
          </TouchableOpacity>
        )}
        keyExtractor={(item) => item.name.toString()}
        estimatedItemSize={20}
      />
    </SafeAreaView>
  );
};

export default SearchStep2;
