import { View, Text, SafeAreaView, Alert } from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import { Ionicons } from "@expo/vector-icons";
import * as Haptics from "expo-haptics";
import BackButton from "../../components/Buttons/BackButton";
import {
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  INPUT_CLASS,
  TEXT_AREA_CLASS,
} from "../../styles/constants";
import PageSubTitle from "../../components/Titles/PageSubTitle";
import PrimaryButton from "../../components/Buttons/PrimaryButton";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import { TextInput } from "react-native";
import { addDoc, collection } from "firebase/firestore";
import { auth, db } from "../../../config/firebase";
import moment from "moment";
import { useTranslation } from "react-i18next"; // Import useTranslation hook

export default function SignalBug({ navigation, route }) {
  const MAX_DESC_LENGTH = 500;
  const [isLoading, setIsLoading] = useState(false);
  const [desc, setDesc] = useState("");
  const { t } = useTranslation(); // Initialize useTranslation hook

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: t("signal_bug.header_title"), // Use translation here
    });
  }, [navigation, t]);

  useEffect(() => {
    navigation
      .getParent()
      ?.setOptions({ tabBarStyle: { display: "none" }, tabBarVisible: false });
    return () =>
      navigation
        .getParent()
        ?.setOptions({ tabBarStyle: undefined, tabBarVisible: undefined });
  }, [navigation]);

  const validate = async () => {
    setIsLoading(true);

    if (desc === "") {
      setIsLoading(false);
      return;
    }

    try {
      await addDoc(collection(db, "bugs"), {
        bug: desc,
        userId: auth?.currentUser ? auth.currentUser.uid : null,
        createdAt: moment().format(),
      });

      Alert.alert(
        t("signal_bug.success_alert_title"), // Use translation here
        t("signal_bug.success_alert_message"), // Use translation here
        [{ text: "OK", onPress: () => navigation.goBack() }]
      );
    } catch (e) {
      Alert.alert(t("signal_bug.error_alert_title"), `${e}`);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <SafeAreaView className="flex-1 bg-white">
      <KeyboardAwareScrollView
        className="p-4"
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        scrollEnabled
        behavior="padding"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
      >
        <View className="mt-5">
          <Text
            className="text-sm mb-2 ml-1"
            style={{ fontFamily: "Inter_400Regular" }}
          >
            {t("signal_bug.message_label")} {/* Use translation here */}
          </Text>
          <TextInput
            textAlignVertical="top"
            style={{ fontFamily: "Inter_500Medium" }}
            multiline
            className={TEXT_AREA_CLASS}
            placeholder={t("signal_bug.placeholder_message")} // Use translation here
            placeholderTextColor={"gray"}
            value={desc}
            maxLength={MAX_DESC_LENGTH}
            onChangeText={(val) => {
              setDesc(val);
            }}
          />
          <Text
            className="text-xs text-gray-500 mt-2 ml-1 self-end"
            style={{ fontFamily: "Inter_300Light" }}
          >
            {MAX_DESC_LENGTH - desc.length}{" "}
            {t("signal_bug.remaining_characters")} {/* Use translation here */}
          </Text>
        </View>
        <Text
          className="mt-10 text-xs leading-5 text-gray-500"
          style={{ fontFamily: "Inter_300Light" }}
        >
          {t("signal_bug.feedback_message")} {/* Use translation here */}
        </Text>

        <View className="flex-1 mt-10 w-full">
          <PrimaryButton
            text={t("signal_bug.send_button")} // Use translation here
            isLoading={isLoading}
            onPress={validate}
          />
        </View>
      </KeyboardAwareScrollView>
    </SafeAreaView>
  );
}
